import numpy as np
import matplotlib.pyplot as plt

def generate_mackey_glass(length=2000, tau=17, delta_t=0.1):
    """
    Mackey-Glass方程式に基づくカオス時系列データを生成する関数
    
    Parameters:
        length (int): 生成するデータ点数
        tau (int): 遅延パラメータ (17以上でカオスになります)
        delta_t (float): 時間刻み
    """
    # カオス生成のための標準的なパラメータ
    beta = 0.2
    gamma = 0.1
    n = 10
    
    # 履歴を保持するためのバッファ（初期値はランダムではなく固定値で開始）
    history_len = int(tau / delta_t)
    history = [1.2] * history_len
    
    timeseries = []
    
    # 初期過渡応答（トランジェント）を捨てるための予備計算
    warmup = 1000
    total_steps = warmup + length
    
    x = 1.2 # 初期値
    
    print(f"データを生成中... ({length} ステップ)")
    
    for i in range(total_steps):
        # 過去の値 x(t - tau) を取得
        x_tau = history.pop(0)
        
        # Mackey-Glass微分方程式 (の離散化)
        # dx/dt = beta * x_tau / (1 + x_tau^n) - gamma * x
        dx = (beta * x_tau / (1 + x_tau**n) - gamma * x) * delta_t
        x += dx
        
        # 履歴を更新
        history.append(x)
        
        # ウォームアップが終わったら記録開始
        if i >= warmup:
            timeseries.append(x)
            
    return np.array(timeseries)

# --- メイン処理 ---

# 1. データを生成
data_length = 2000
mg_data = generate_mackey_glass(length=data_length)

# 2. 扱いやすいように 0.0 ～ 1.0 の範囲に正規化（スケーリング）
#    ※ニューラルネットやリザバー計算では正規化しておくのが定石です
min_val = np.min(mg_data)
max_val = np.max(mg_data)
mg_data_normalized = (mg_data - min_val) / (max_val - min_val)

# 3. テキストファイルとして保存
filename = "mackey_glass.txt"
np.savetxt(filename, mg_data_normalized, fmt='%.6f')

print(f"完了: '{filename}' に保存しました。")

# 4. 確認用プロット（最初の500点だけ表示）
plt.figure(figsize=(12, 4))
plt.plot(mg_data_normalized[:500], label="Mackey-Glass Data (Normalized)")
plt.title("Generated Chaos Time Series (First 500 steps)")
plt.xlabel("Step")
plt.ylabel("Value")
plt.legend()
plt.grid(True)
plt.show()